package furny.ga.logger;

import furny.ga.FurnLayoutIndividual;
import ga.core.algorithm.util.PopulationUtil;
import ga.core.individual.IndividualList;
import ga.core.individual.population.IPopulation;
import ga.core.logging.IGALogger;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * GA logger that outputs informations to a CSV file.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public class GACSVLogger implements IGALogger<FurnLayoutIndividual> {
  private static final String DEFAULT_FILE = "fitness.csv";
  private final List<double[]> data = new ArrayList<double[]>();

  private final String csvFileName;

  /**
   * Creates a new CSV logger that writes to the default file.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public GACSVLogger() {
    csvFileName = DEFAULT_FILE;
  }

  /**
   * Creates a new CSV logger that writes to the given file.
   * 
   * @param csvFileName
   *          The CSV file name.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public GACSVLogger(final String csvFileName) {
    this.csvFileName = csvFileName;
  }

  @Override
  public void evaluationStarted() {
    System.out.println("Evaluation started on " + new Date());
  }

  @Override
  public void populationInitiated(final int generation,
      final IPopulation<FurnLayoutIndividual> population) {
  }

  @Override
  public void individualsSelected(final int generation,
      final IndividualList<FurnLayoutIndividual> list) {
  }

  @Override
  public void individualsCrossed(final int generation,
      final IndividualList<FurnLayoutIndividual> list) {
  }

  @Override
  public void individualsMutated(final int generation,
      final IndividualList<FurnLayoutIndividual> list) {
  }

  @Override
  public void individualsInserted(final int generation,
      final IndividualList<FurnLayoutIndividual> list,
      final IPopulation<FurnLayoutIndividual> population) {
  }

  @Override
  public void allIndividualsEvaluated(final int generation,
      final IPopulation<FurnLayoutIndividual> population) {
    data.add(PopulationUtil.getMinMeanMaxFitness(population));

    System.out.println("Writing CSV file");

    // write the file in between
    writeCSV(data.toArray(new double[data.size()][]), csvFileName);
  }

  @Override
  public void individualSelectedForEvaluation(final int generation,
      final FurnLayoutIndividual individual) {
  }

  @Override
  public void individualEvaluated(final int generation,
      final FurnLayoutIndividual individual) {

  }

  @Override
  public void exit() {
    System.out.println("Writing CSV file");

    // write the file on exit
    writeCSV(data.toArray(new double[data.size()][]), csvFileName);

    System.out.println("Logger closed");

  }

  /**
   * Writes the data array to the CSV file.
   * 
   * @param data
   *          The data array.
   * @param filename
   *          The CSV file.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  private static void writeCSV(final double[][] data, final String filename) {
    final File csvFile = new File(filename);
    BufferedWriter bw = null;
    try {
      bw = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(
          csvFile)));

      bw.write("i;fit_min;fit_mean;fit_max\n");

      for (int i = 0; i < data.length; i++) {
        try {
          bw.write(String.format("%d;%,.2f;%,.2f;%,.2f;\n", i, data[i][0],
              data[i][1], data[i][2]));
        } catch (final Exception e) {
          e.printStackTrace();
        }

      }

    } catch (final IOException e) {
      e.printStackTrace();
    } finally {
      try {
        bw.close();
      } catch (final Exception e) {
      }
    }
  }
}
